; Shadowrun SNES Mouse patch
; by rainwarrior (Brad Smith)
; https://rainwarrior.ca
; 2022-03-07
;
; Based on an emulator script from Gamachara:
; https://github.com/Gamachara/SNES-Shadowrun-Mouse-Script
;
; Version 1

; Controls:
;   Connect a SNES mouse to the second controller port.
;   Use a controller with your left hand.
;
; The controller works as normal except for L/R which now affect the mouse.
;
; Left/right mouse clicks map to B and A normally.
; Hold L on the controller for alternate button use.
;   On the MAIN GAME SCREEN or in THE MATRIX:
;     L+Left = START (Status)
;     L+Right = X (Spell)
;   With the ACTION GLOVE active:
;     L+Left = L (Examine)
;     L+Right = R (Open)
;   In a menu:
;     L+Left = L (Page Up)
;     L+Right = R (Page Up)
;
; Press R on the controller to cycle mouse sensitivity. (Mid/High/Low)
;
; The Y button (quick Inventory) can't be activated with the mouse, but you can still reach Items through the START menu.
; The game can be played without ever needing to use buttons on the right half of the controller.

VERSION = 1
YEAR = 2022

.setcpu "65816"

.import ADDR_INPUT   : far ; controller 1 readout: LDA $4218 STA VAR_INPUT (should be in one place only)
.import ADDR_INIT    : far ; long JMP at end of Shadowrun's initialization stub
.import ADDR_INITJMP : far ; destination of long JMP at end of the initialization stub
.import ADDR_COPY    : far ; Title screen copyright string
.import ADDR_EMPTY   : far ; empty space in ROM to insert patch code

.import VAR_FREEHIGH : far ; 6 unused bytes in RAM
.import VAR_FREELOW  : abs ; 2 unused bytes in LowRAM
.import VAR_INPUT    : abs ; controller input variable storage
.import VAR_CX       : abs ; cursor X
.import VAR_CY       : abs ; cursor Y
.import VAR_CURSOR   : abs ; cursor state (0 off, -1 hand, -2 gun, -3 spell)
.import VAR_DIALOG   : abs ; dialog state (0 off, -1 in menu or dialog)

.import __INPUT_SIZE__
.import __INIT_SIZE__
.import __COPY_SIZE__
.import __EMPTY_SIZE__

; converts LoROM program address to PRG-ROM address
.define LoROM(addr_) (((addr_)&$007FFF)|(((addr_)&$3F0000)>>1))

; convert memory addresses to ROM locations
ROM_INPUT  = LoROM ADDR_INPUT
ROM_INIT   = LoROM ADDR_INIT
ROM_COPY   = LoROM ADDR_COPY
ROM_EMPTY  = LoROM ADDR_EMPTY

;
; IPS file structure
;

.macro IPS_OFFSET value_
	.assert (value_)<=$FFFFFF,error,"IPS offset range error"
	.byte <((value_)>>16)
	.byte >(value_)
	.byte <(value_)
.endmacro

.macro IPS_SIZE value_
	.byte >(value_)
	.byte <(value_)
.endmacro

.segment "HEADER"
.byte "PATCH"

.segment "INPUT_C"
IPS_OFFSET ROM_INPUT
IPS_SIZE __INPUT_SIZE__

.segment "INIT_C"
IPS_OFFSET ROM_INIT
IPS_SIZE   __INIT_SIZE__

.segment "COPY_C"
IPS_OFFSET ROM_COPY
IPS_SIZE   __COPY_SIZE__

.segment "EMPTY_C"
IPS_OFFSET ROM_EMPTY
IPS_SIZE   __EMPTY_SIZE__

.segment "FOOTER"
.byte "EOF"

;
; Patch chunk contents
;

; variables
mouseread_y = VAR_FREELOW+0 ; 3rd mouse report byte (Y)
mouseread_x = VAR_FREELOW+1 ; 4th mouse report byte (X)
mouseread_s = VAR_FREEHIGH+0 ; 2nd mouse report byte (buttons, speed, signature)
padh_last   = VAR_FREEHIGH+1 ; last state of controller buttons second byte: AXLR0000
mousepos_x  = VAR_FREEHIGH+2 ; absolute position of mouse cursor
mousepos_y  = VAR_FREEHIGH+3
mouselock   = VAR_FREEHIGH+4 ; button mode lock (0=unlock, 1=open, 2=main-L, 3=alt-L)
padlock     = VAR_FREEHIGH+5 ; using dpad can disable mouse override until it moves again


.segment "INPUT"
	; replaces LDA $4218 STA VAR_INPUT
	jsl f:input
	nop
	nop

.segment "INIT"
	; replaces jmp f:ADDR_INITJMP at end of initialization stub
	jmp f:init

.segment "COPY"
;      "(C) 1993 DATAEAST USA, Inc."0
.byte "   Mouse patch v"; (year)  "
.byte '0' + VERSION
.byte " ("
.byte '0' + ((YEAR/1000) .mod 10)
.byte '0' + ((YEAR/ 100) .mod 10)
.byte '0' + ((YEAR/  10) .mod 10)
.byte '0' + ((YEAR/   1) .mod 10)
.byte ")   ",0

.segment "EMPTY"

; Take over just after the initialization stub to initialize the mouse.
; Note: ensure $4200 auto joypad poll is disabled coming into this routine.
init:
	php
	sep #$30
	.a8
	.i8
	pha
	phx
	phy
	; try to set mouse to medium speed to initialize it
	ldx #4
	:
		jsr mouse_speed ; strobe mouse and and cycle speed
		jsr mouse_bits ; read 16 bits
		lda a:mouseread_y
		and #$30 ; check current speed setting
		cmp #$10 ; look for medium speed
		php
		jsr mouse_bits ; read the other 16 bits
		plp
		beq :+
		dex ; retry 4x
		bne :-
	:
	; mouse is now initialized (if connected)
	; center starting mouse cursor
	lda #128
	sta f:mousepos_x
	lda #120
	sta f:mousepos_y
	; initialize other variables
	lda #0
	sta f:padh_last
	sta f:mouselock
	sta f:padlock
	ply
	plx
	pla
	plp
	jmp f:ADDR_INITJMP ; go where the initialization stub was supposed to go

; Replaces copying the 16-bit auto-read of controller 1, and augments it with a mouse read.
input:
	; replaced code
	.a16
	.i16
	lda a:$4218
	sta a:VAR_INPUT
	php
	sep #$30
	.a8
	.i8
	phx
	phy
	; read mouse report 2nd byte (1st byte is ignored, should be 00)
	lda a:$421A
	sta f:mouseread_s
	; check for mouse signature
	and #$0F
	cmp #$01
	beq :+
		jmp @end ; no mouse = skip
	:
	; read Y/X bytes
	jsr mouse_bits
	; R on controller cycles mouse speed
	lda a:VAR_INPUT+0
	eor f:padh_last
	and a:VAR_INPUT+0
	and #$10 ; R just pressed
	beq :+
		jsr mouse_speed
	:
	; remember last-press of controller AXLR
	lda a:VAR_INPUT+0
	sta f:padh_last
	; clear reported L/R presses so we can override them with the mouse
	and #$CF
	sta a:VAR_INPUT+0
	; if mouse buttons are released, unlock button mode
	lda f:mouseread_s
	and #$C0
	bne :+
		sta f:mouselock ; A=0 unlock
		jmp @click_end
	:
	; if button mode is locked, select it now
	lda f:mouselock
	beq @unlocked ; 0 = unlocked
	cmp #2
	beq @main_L ; 2 = main_L
	cmp #3
	beq @alt_L ; 3 = alt_L
	jmp @open_L ; 1 or other = open_L
@unlocked:
	; mouse buttons (mode selected with controller L)
	lda f:padh_last
	and #$20 ; L
	bne @closed_L
@open_L:
	lda #1
	sta f:mouselock
	; if L is not held, clicks always map to B/A
	lda f:mouseread_s
	and #$40 ; left button
	beq :+
		lda #$80 ; B
		tsb a:VAR_INPUT+1
	:
	lda f:mouseread_s
	and #$80 ; right button
	beq :+
		lda #$80 ; A
		tsb a:VAR_INPUT+0
	:
	jmp @click_end
@closed_L:
	lda a:VAR_DIALOG ; if in a menu, not alt
	bne :+
		lda a:VAR_CURSOR
		beq @alt_L ; mouse not active = alt
		cmp #$FD ; spell casting = alt
		beq @alt_L
	:
@main_L:
	lda #2
	sta f:mouselock
	; if L is held, clicks map to L/R
	lda f:mouseread_s
	and #$40
	beq :+
		lda #$20 ; L
		tsb a:VAR_INPUT+0
	:
	lda f:mouseread_s
	and #$80
	beq :+
		lda #$10 ; R
		tsb a:VAR_INPUT+0
	:
	jmp @click_end
@alt_L:
	lda #3
	sta f:mouselock
	; if L is held, but not in menu or active mouse/gun, clicks map to START/X
	lda f:mouseread_s
	and #$40
	beq :+
		lda #$10 ; START
		tsb a:VAR_INPUT+1
	:
	lda f:mouseread_s
	and #$80
	beq :+
		lda #$40 ; X
		tsb a:VAR_INPUT+0
	:
	;jmp @click_end
@click_end:
	; cursor motion and jake motion when cursor active
	lda a:VAR_CURSOR
	cmp #$FD
	bcs :+
		jmp @cursor_end
	:
	; if mouse not moving but d-pad pressed, allow d-pad to take over
	lda a:mouseread_x
	ora a:mouseread_y
	and #$7F
	bne @mouse_moved ; moving the mouse clears padlock
	lda f:padlock
	beq :+ ; padlocked = let d-pad controls override the mouse position
		lda a:VAR_CX
		sta f:mousepos_x
		lda a:VAR_CY
		sta f:mousepos_y
		jmp @cursor_end ; padlock on = d-pad controls instead
	:
	lda a:VAR_INPUT+1
	and #$0F ; d-pad
	beq :+
		lda #$FF
		sta f:padlock ; if d-pad pressed but mouse is stationary: activate padlock
		jmp @cursor_end
	:
	; not padlocked, just let mouse position override cursor
	lda f:mousepos_x
	sta a:VAR_CX
	lda f:mousepos_y
	sta a:VAR_CY
	jmp @cursor_end
@mouse_moved:
	lda #0
	sta f:padlock ; resume mouse control if locked by d-pad
	; move the mouse horizontal (clamp 3-255)
	lda a:mouseread_x ; mouse X delta
	bmi @mousex_left
@mousex_right:
	clc
	adc f:mousepos_x
	bcs :+
	cmp #$F0
	bcc :++
	:
		lda #$EF
	:
	jmp @mousex_end
@mousex_left:
	and #$7F
	eor #$FF
	sec
	adc f:mousepos_x
	bcc :+
	cmp #3
	bcs :++
	:
		lda #3
	:
@mousex_end:
	sta f:mousepos_x
	sta a:VAR_CX
	; move the mouse vertical (clamp 3-223)
	lda a:mouseread_y ; mouse Y delta
	bmi @mousey_up
@mousey_down:
	clc
	adc f:mousepos_y
	bcs :+
	cmp #$CF
	bcc :++
	:
		lda #$CE
	:
	jmp @mousey_end
@mousey_up:
	and #$7F
	eor #$FF
	sec
	adc f:mousepos_y
	bcc :+
	cmp #3
	bcs :++
	:
		lda #3
	:
@mousey_end:
	sta f:mousepos_y
	sta a:VAR_CY
@cursor_end:
	; resume
@end:
	ply
	plx
	plp
	lda a:VAR_INPUT
	rtl

mouse_speed:
	.a8
	lda #1
	sta a:$4016 ; controller strobe on
	lda a:$4017 ; cycle mouse speed
	stz a:$4016 ; controller strobe off
	rts

mouse_bits: ; reads 16-bits of mouse port to mouseread_y/mouseread_x
	.a8
	ldy #16
@read_bit:
	lda a:$4017
	lsr
	rol a:mouseread_x
	rol a:mouseread_y
	nop ; compensate for lack of indexing, vs MOUSE.X65 reads
	dey
	bne @read_bit
	rts
